/*
 * Decompiled with CFR 0.152.
 */
package com.neptunelabs.fsiframework.cache;

import com.neptunelabs.fsiframework.cache.CacheLoad;
import com.neptunelabs.fsiframework.cache.CacheType;
import com.neptunelabs.fsiframework.io.MemoryManager;
import com.neptunelabs.fsiframework.logging.FSILogger;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantReadWriteLock;

public final class BlockCache {
    private volatile int maxCaches;
    private final long maxMemory;
    private volatile int caches = 0;
    private final ByteBuffer[] cacheArray;
    private final int[][] cacheAllocationArray;
    private final int[] cacheWriteCursor;
    private volatile int lastCacheWritten = 0;
    private final ReentrantReadWriteLock baseLock = new ReentrantReadWriteLock();
    private final ReentrantReadWriteLock.ReadLock readLock = this.baseLock.readLock();
    private final ReentrantReadWriteLock.WriteLock writeLock = this.baseLock.writeLock();
    private final Map<String, CacheObject> cacheKeyMap = new ConcurrentHashMap<String, CacheObject>();
    private static final int MAX_SIZE_PER_CACHE = 0x7FFFF000;
    private static final int MIN_HEAP = 0x40000000;
    private static final int MIN_OBJECT_SIZE = 10;
    private static final int BLOCK_SIZE = 4096;
    private final AtomicLong touchCounter = new AtomicLong(0L);
    private long cleanUpCounter = 0L;
    private long cleanUpTime = 0L;
    private long reserveCounter = 0L;
    private long reserveTime = 0L;
    private long deleteCounter = 0L;
    private long deleteTime = 0L;
    private int statMaxObjects = 0;
    private final boolean direct;
    private boolean enabled = false;
    private boolean oomError = false;

    public BlockCache(FSILogger logger, long calcSize, boolean direct, boolean enabled) {
        this.direct = direct;
        this.maxCaches = (int)Math.ceil((double)calcSize / 2.147479552E9);
        this.maxMemory = calcSize;
        this.cacheArray = new ByteBuffer[this.maxCaches];
        this.cacheAllocationArray = new int[this.maxCaches][];
        this.cacheWriteCursor = new int[this.maxCaches];
        this.allocateNextCache(this.caches);
        this.enabled = enabled;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private synchronized boolean allocateNextCache(int currentCaches) {
        boolean successfull = true;
        if (this.caches >= this.maxCaches) return false;
        if (currentCaches != this.caches) return false;
        this.writeLock.lock();
        try {
            this.cacheWriteCursor[this.caches] = 0;
            int maxAlloc = this.maxMemory < 0x7FFFF000L ? (int)this.maxMemory / 4096 * 4096 : 0x7FFFF000;
            long freeMem = MemoryManager.getFreeHeapSpace();
            if (freeMem < 0x40000000L || freeMem < (long)maxAlloc) {
                this.maxCaches = this.caches;
                successfull = false;
            }
            if (!successfull) return successfull;
            try {
                this.cacheArray[this.caches] = this.direct ? ByteBuffer.allocateDirect(maxAlloc) : ByteBuffer.allocate(maxAlloc);
                int blocks = maxAlloc / 4096;
                this.cacheAllocationArray[this.caches] = new int[blocks];
                ++this.caches;
                return successfull;
            }
            catch (OutOfMemoryError oome) {
                this.oomError = true;
                successfull = false;
            }
            return successfull;
        }
        finally {
            this.writeLock.unlock();
        }
    }

    public boolean hasOOME() {
        return this.oomError;
    }

    public boolean isDirect() {
        return this.direct;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CacheLoad loadObject(String key) {
        this.readLock.lock();
        CacheLoad result = null;
        try {
            CacheObject cacheObj;
            if (this.enabled && (cacheObj = this.cacheKeyMap.get(key)) != null) {
                ByteBuffer cache = this.cacheArray[cacheObj.cacheNum].slice();
                result = new CacheLoad(cacheObj.type, new byte[cacheObj.realLength]);
                cache.position(cacheObj.startBlock * 4096);
                cache.get(result.data);
                cacheObj.touchPoint = this.touchCounter.getAndIncrement();
            }
        }
        finally {
            this.readLock.unlock();
        }
        return result;
    }

    public boolean hasObject(String key) {
        return this.cacheKeyMap.containsKey(key);
    }

    public void clear() {
        this.cleanUp(0, 100);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean saveObject(String key, CacheLoad cacheLoad) {
        boolean result = false;
        this.writeLock.lock();
        try {
            if (this.enabled && cacheLoad != null && cacheLoad.data.length > 10) {
                CacheObject cacheObj = this.cacheKeyMap.get(key);
                if (cacheObj != null) {
                    this.deleteObject(cacheObj);
                }
                if ((cacheObj = this.reserveFreeSpace(cacheLoad.data.length)) == null) {
                    for (int c = 0; c < 5; ++c) {
                        if (!this.allocateNextCache(this.caches)) {
                            this.cleanUp(cacheLoad.data.length, 10);
                        }
                        if ((cacheObj = this.reserveFreeSpace(cacheLoad.data.length)) != null) break;
                    }
                }
                if (cacheObj != null) {
                    ByteBuffer cache = this.cacheArray[cacheObj.cacheNum].slice();
                    cache.position(cacheObj.startBlock * 4096);
                    cache.put(cacheLoad.data);
                    cacheObj.key = key;
                    cacheObj.type = cacheLoad.type;
                    this.cacheKeyMap.put(key, cacheObj);
                    result = true;
                }
            }
        }
        finally {
            this.writeLock.unlock();
        }
        return result;
    }

    public boolean deleteKey(String key) {
        boolean result = false;
        CacheObject cacheObj = this.cacheKeyMap.get(key);
        if (cacheObj != null) {
            this.deleteObject(cacheObj);
            cacheObj = null;
            result = true;
        }
        return result;
    }

    public boolean deleteKeys(Set<String> keys) {
        boolean result = true;
        if (this.enabled) {
            for (String key : keys) {
                result |= this.deleteKey(key);
            }
        }
        return result;
    }

    public long getMaximumCacheSize() {
        if (this.enabled) {
            return (long)this.maxCaches * 0x7FFFF000L;
        }
        return 0L;
    }

    public long getAllocatedCacheSize() {
        if (this.enabled) {
            long size = 0L;
            for (int c = 0; c < this.caches; ++c) {
                size += (long)this.cacheArray[c].capacity();
            }
            return size;
        }
        return 0L;
    }

    public long getUsedCacheSize() {
        long result = 0L;
        for (CacheObject co : this.cacheKeyMap.values()) {
            result += (long)((co.endBlock - co.startBlock + 1) * 4096);
        }
        return result;
    }

    public void setEnabled(boolean state) {
        this.enabled = state;
    }

    public boolean isEnabled() {
        return this.enabled;
    }

    public long getObjectSize(String key) {
        CacheObject co;
        if (this.enabled && (co = this.cacheKeyMap.get(key)) != null) {
            return co.realLength;
        }
        return 0L;
    }

    public String toStatisticsString() {
        StringBuilder sb = new StringBuilder();
        int count = 0;
        int min = Integer.MAX_VALUE;
        int max = Integer.MIN_VALUE;
        long sum = 0L;
        for (CacheObject co : this.cacheKeyMap.values()) {
            int len = co.realLength;
            sum += (long)len;
            min = Math.min(len, min);
            max = Math.max(len, max);
            ++count;
        }
        int avg = count > 0 ? (int)(sum / (long)count) : 0;
        sb.append("Current Objects: " + count + "\n");
        sb.append("Max Objects: " + this.statMaxObjects + "\n\n");
        sb.append("Min Object Size: " + min + "\n");
        sb.append("Max Object Size: " + max + "\n");
        sb.append("Avg Object Size: " + avg + "\n");
        sb.append("Total Object Size: " + sum + "\n\n");
        sb.append("Cache Blocks: " + this.caches + "\n");
        for (int c = 0; c < this.caches; ++c) {
            int[] caa = this.cacheAllocationArray[c];
            int free = 0;
            int used = 0;
            for (int i = 0; i < this.cacheArray[c].capacity() / 4096; ++i) {
                if (caa[i] == 0) {
                    ++free;
                    continue;
                }
                if (caa[i] == 0) continue;
                ++used;
            }
            sb.append("Cache Block " + (c + 1) + " Used Blocks: " + used + "\n");
            sb.append("Cache Block " + (c + 1) + " Free Blocks: " + free + "\n");
        }
        return sb.toString();
    }

    public void destroy() {
        int i;
        this.enabled = false;
        if (this.cacheArray != null && this.cacheArray.length > 0) {
            for (i = 0; i < this.cacheArray.length; ++i) {
                this.cacheArray[i] = null;
            }
        }
        if (this.cacheAllocationArray != null && this.cacheAllocationArray.length > 0) {
            for (i = 0; i < this.cacheAllocationArray.length; ++i) {
                this.cacheAllocationArray[i] = null;
            }
        }
    }

    public Set<String> getKeys() {
        return this.cacheKeyMap.keySet();
    }

    public int size() {
        return this.cacheKeyMap.size();
    }

    public long getCleanupCounter() {
        return this.cleanUpCounter;
    }

    public long getCleanupTime() {
        return this.cleanUpTime;
    }

    public long getReserveCounter() {
        return this.reserveCounter;
    }

    public long getReserveTime() {
        return this.reserveTime;
    }

    public long getDeleteCounter() {
        return this.deleteCounter;
    }

    public long getDeleteTime() {
        return this.deleteTime;
    }

    private void deleteObject(CacheObject cacheObj) {
        if (cacheObj != null) {
            int[] cache = this.cacheAllocationArray[cacheObj.cacheNum];
            long startTime = System.currentTimeMillis();
            this.cacheKeyMap.remove(cacheObj.key);
            for (int a = cacheObj.startBlock; a <= cacheObj.endBlock; ++a) {
                cache[a] = 0;
            }
            this.cacheWriteCursor[cacheObj.cacheNum] = cacheObj.startBlock;
            this.lastCacheWritten = cacheObj.cacheNum;
            ++this.deleteCounter;
            this.deleteTime += System.currentTimeMillis() - startTime;
        }
    }

    private CacheObject reserveFreeSpace(int size) {
        int usedCache = -1;
        int startBlock = -1;
        int endBlock = -1;
        int neededBlocks = (int)Math.ceil((float)size / 4096.0f);
        if (this.lastCacheWritten >= this.caches) {
            this.lastCacheWritten = 0;
        }
        int cacheNum = this.lastCacheWritten;
        long startTime = System.currentTimeMillis();
        for (int c = 0; c < this.caches; ++c) {
            int[] cache = this.cacheAllocationArray[cacheNum];
            startBlock = -1;
            endBlock = -1;
            int foundBlocks = 0;
            int cachePosition = this.cacheWriteCursor[cacheNum];
            int maxBlocks = cache.length - 1;
            if (cachePosition >= maxBlocks) {
                cachePosition = 0;
            }
            for (int a = 0; a < maxBlocks; ++a) {
                if (cache[cachePosition] != 0) {
                    int cP = cache[cachePosition];
                    a += cP - 1;
                    if ((cachePosition += cP) >= maxBlocks) {
                        cachePosition = 0;
                    }
                    foundBlocks = 0;
                    startBlock = -1;
                    continue;
                }
                ++foundBlocks;
                if (startBlock == -1) {
                    startBlock = cachePosition;
                }
                if (foundBlocks == neededBlocks) {
                    endBlock = cachePosition;
                    usedCache = cacheNum;
                    break;
                }
                if (++cachePosition >= maxBlocks) {
                    cachePosition = 0;
                    foundBlocks = 0;
                    startBlock = -1;
                    continue;
                }
                if (cachePosition + neededBlocks < maxBlocks) continue;
                a += maxBlocks - cachePosition;
                cachePosition = 0;
                foundBlocks = 0;
                startBlock = -1;
            }
            if (startBlock != -1 && endBlock != -1) break;
            if (++cacheNum < this.caches) continue;
            cacheNum = 0;
        }
        if (startBlock != -1 && endBlock != -1) {
            this.cacheWriteCursor[usedCache] = endBlock + 1;
            int[] caa = this.cacheAllocationArray[usedCache];
            int u = neededBlocks;
            for (int r = startBlock; r <= endBlock; ++r) {
                caa[r] = u--;
            }
            CacheObject cacheObj = new CacheObject();
            cacheObj.cacheNum = usedCache;
            cacheObj.startBlock = startBlock;
            cacheObj.endBlock = endBlock;
            cacheObj.realLength = size;
            cacheObj.touchPoint = this.touchCounter.getAndIncrement();
            ++this.reserveCounter;
            this.reserveTime += System.currentTimeMillis() - startTime;
            this.lastCacheWritten = usedCache;
            return cacheObj;
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean cleanUp(int minSize, int percent) {
        boolean successful = false;
        this.writeLock.lock();
        try {
            long startTime = System.currentTimeMillis();
            this.statMaxObjects = Math.max(this.statMaxObjects, this.cacheKeyMap.size());
            ArrayList<CacheObject> cList = new ArrayList<CacheObject>(this.cacheKeyMap.values());
            Collections.sort(cList, new CacheObjectComparator());
            int cut = (int)Math.ceil((float)(percent * cList.size()) / 100.0f);
            int deleted = 0;
            for (CacheObject co : cList) {
                this.deleteObject(co);
                if (++deleted <= cut) continue;
                break;
            }
            successful = true;
            ++this.cleanUpCounter;
            this.cleanUpTime += System.currentTimeMillis() - startTime;
        }
        finally {
            this.writeLock.unlock();
        }
        return successful;
    }

    private final class CacheObjectComparator
    implements Comparator<CacheObject> {
        @Override
        public int compare(CacheObject o1, CacheObject o2) {
            if (o1.touchPoint < o2.touchPoint) {
                return -1;
            }
            if (o1.touchPoint > o2.touchPoint) {
                return 1;
            }
            return 0;
        }
    }

    private class CacheObject {
        int cacheNum;
        int startBlock;
        int endBlock;
        int realLength;
        long touchPoint = 0L;
        String key;
        CacheType type;
    }
}

